/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Collections;
using System.ComponentModel;
using System.Globalization;

using Borland.Eco.ObjectRepresentation;
using Borland.Eco.Support.Deriver;
using Borland.Eco.Services;
using Borland.Eco.Subscription;
using Borland.Eco.UmlRt;

namespace Borland.Eco.Handles
{
	///<summary>
	///Enumeration used to filter members on visibility.
	///</summary>
	public enum MemberVisibility
	{
		///<summary>All members are shown.</summary>
		AllMembers,
		///<summary>Public + protected members are shown.</summary>
		ProtectedOrHigher,
		///<summary>Only public members are shown.</summary>
		PublicOnly
	}
	/// <summary>
	/// This interface is part of the internal implementation of the databindable IList
	/// implemented by the Eco handles. It should not be used directly.
	/// </summary>
	public interface INotifyOnIdle
	{
		void NotifyOnIdle();
	}
	/// <summary>
	/// This interface is part of the internal implementation of the databindable IList
	/// implemented by the Eco handles. It should not be used directly.
	/// </summary>
	public interface IRenderingContext
	{
		IEcoServiceProvider ServiceProvider {get;}
	}
	/// <summary>
	/// This interface is part of the internal implementation of the databindable IList
	/// implemented by the Eco handles. It should not be used directly.
	/// </summary>
	public interface ITuplePropertyRenderingContext: IRenderingContext, IElementProvider
	{
		void PropertyOutOfDate(); // Callback
	}
	/// <summary>
	/// This interface is part of the internal implementation of the databindable IList
	/// implemented by the Eco handles. It should not be used directly.
	/// </summary>
	public interface ITupleRenderingContext: IRenderingContext
	{
		RenderedTupleDescriptor TupleDescriptor { get; }
		void PartOutOfDate(); // Callback
	}
	/// <summary>
	/// This interface is part of the internal implementation of the databindable IList
	/// implemented by the Eco handles. It should not be used directly.
	/// </summary>
	public interface ITupleListRenderingContext: ITupleRenderingContext, ISubscribableElementProvider
	{
	}

	///<summary>
	///Helper class with useful routines. This class contains only static methods and cannot be instantiated.
	///</summary>
	public sealed class HandleUtils
	{
		private HandleUtils() {}
		///<summary>
		///Returns the type of the <paramref name="classifier"/> or the type of its elements, if it happens to be a collection type.
		///</summary>
		public static IClassifier ItemType(IClassifier classifier)
		{
			ICollectionType ct = classifier as ICollectionType;
			return ct != null ? ct.ElementType : classifier;
		}
	}
	/// <summary>
	/// This class is part of the internal implementation of the databindable Ilist
	/// implemented by the Eco handles. It should not be used directly.
	/// </summary>
	public sealed class ItemStaticContext: IStaticContext
	{
		private readonly IStaticContext m_AdaptedContext;

		public ItemStaticContext(IStaticContext adaptedContext)
		{
			m_AdaptedContext = adaptedContext;
		}

		IEcoTypeSystem IStaticContext.TypeSystem
		{
			get { return m_AdaptedContext.TypeSystem; }
		}

		IOclTypeService IStaticContext.OclTypeService
		{
			get { return m_AdaptedContext.OclTypeService; }
		}

		IOclPsTypeService IStaticContext.OclPsTypeService
		{
			get { return m_AdaptedContext.OclPsTypeService; }
		}

		IActionLanguageTypeService IStaticContext.ActionLanguageTypeService
		{
			get { return m_AdaptedContext.ActionLanguageTypeService; }
		}

		IExternalVariableList IStaticContext.VariableList
		{
			get { return m_AdaptedContext.VariableList; }
		}

		IClassifier IStaticContext.StaticUmlType
		{
			get { return HandleUtils.ItemType(m_AdaptedContext.StaticUmlType); }
		}
	}

	public sealed class DeriveEventArgs: EventArgs
	{
		private readonly string m_Name;
		private readonly IElement m_RootElement;
		private IElement m_ResultElement;
		private readonly ISubscriber m_ValueChangeSubscriber;
		private readonly ISubscriber m_ResubscribeSubscriber;
		private readonly IEcoServiceProvider m_ServiceProvider;
		private readonly IStaticContext m_StaticContext;

		public DeriveEventArgs(string name, IEcoServiceProvider serviceProvider, IElement rootElement, ISubscriber valueChangeSubscriber, ISubscriber resubscribeSubscriber, IStaticContext staticContext)
		{
			m_RootElement = rootElement;
			m_ValueChangeSubscriber = valueChangeSubscriber;
			m_ResubscribeSubscriber = resubscribeSubscriber;
			m_ServiceProvider = serviceProvider;
			m_StaticContext = staticContext;
			m_Name = name;
		}

		public IEcoServiceProvider ServiceProvider
		{
			get { return m_ServiceProvider; }
		}

		public IOclService OclService
		{
			get { return m_ServiceProvider.GetEcoService(typeof(IOclService)) as IOclService; }
		}
		/// <summary>
		/// <para>This is the name given to the property added to the binding list.</para>
		/// <para>This name is used when binding to a data aware control.</para>
		/// </summary>
		public string Name
		{
			get { return m_Name; }
		}

		public IStaticContext StaticContext
		{
			get { return m_StaticContext; }
		}

		public IElement RootElement
		{
			get { return m_RootElement; }
		}

		public IElement ResultElement
		{
			set { m_ResultElement = value; }
			get { return m_ResultElement; }
		}

		public ISubscriber ValueChangeSubscriber
		{
			get { return m_ValueChangeSubscriber; }
		}

		public ISubscriber ResubscribeSubscriber
		{
			get { return m_ResubscribeSubscriber; }
		}
	}

	public sealed class ReverseDeriveEventArgs: EventArgs
	{
		private readonly string m_Name;
		private readonly IElement m_RootElement;
		private readonly object m_Value;
		private readonly IEcoServiceProvider m_ServiceProvider;
		private readonly IStaticContext m_StaticContext;

		public ReverseDeriveEventArgs(string name, IEcoServiceProvider serviceProvider, IElement rootElement, IStaticContext staticContext, object value)
		{
			m_RootElement = rootElement;
			m_ServiceProvider = serviceProvider;
			m_StaticContext = staticContext;
			m_Name = name;
			m_Value = value;
		}

		public IEcoServiceProvider ServiceProvider
		{
			get { return m_ServiceProvider; }
		}

		public IOclService OclService
		{
			get { return m_ServiceProvider.GetEcoService(typeof(IOclService)) as IOclService; }
		}

		public string Name
		{
			get { return m_Name; }
		}

		public object Value
		{
			get { return m_Value; }
		}

		public IStaticContext StaticContext
		{
			get { return m_StaticContext; }
		}

		public IElement RootElement
		{
			get { return m_RootElement; }
		}
	}

	// Delegate declaration.
	public delegate void DeriveEventHandler(object sender, DeriveEventArgs e);
	public delegate void ReverseDeriveEventHandler(object sender, ReverseDeriveEventArgs e);
	public delegate void SignalListChangedHandler(ListChangedEventArgs args);

	public sealed class HandleAdapter: ITupleListRenderingContext, INotifyOnIdle
	{
		private readonly ElementHandle m_Handle;
		private readonly RenderedList m_RenderedList;
		private bool m_InQueue;
		private readonly RenderedTupleDescriptor m_TupleDescriptor;

		public HandleAdapter(ElementHandle handle)
		{
			m_Handle = handle;
			m_RenderedList = new RenderedList(this);
			m_TupleDescriptor = new RenderedTupleDescriptor(new ItemStaticContext(handle));
			TupleDescriptor.DescriptorsChanged += new SignalListChangedHandler(m_RenderedList.SignalListChanged);
		}

		public void SetupProperties(ColumnCollection columns, bool addDefaultProperties, bool addDefaultNestings, bool addExternalId, MemberVisibility memberVisibility, NestingCollection nestings, bool designMode)
		{
			TupleDescriptor.SetupProperties(columns, addDefaultProperties, addDefaultNestings, addExternalId, memberVisibility, new DescriptorFactory(m_Handle, nestings, addDefaultNestings, memberVisibility, designMode), designMode);
			m_RenderedList.TupleDescriptorChanged();
			Enqueue();
		}

		public IList List
		{
			get { return m_RenderedList; }
		}

		private void Enqueue()
		{
			if (m_InQueue)
				return;
			DisplayQueue.Add(this);
			m_InQueue = true;
		}

		public object GetRenderedTupleForElement(IElement element)
		{
			return new RenderedTuple(element, new TupleContext(this));
		}

		public void EnsureCurrent()
		{
			m_RenderedList.MakeListsCurrent();
			if (m_InQueue)
			{
				DisplayQueue.Remove(this);
				m_InQueue = false;
			}
		}

		#region INotifyOnIdle implementation
		void INotifyOnIdle.NotifyOnIdle()
		{
			m_RenderedList.MakeListsCurrent();
			m_InQueue = false; // Caller removes from queue
		}
		#endregion

		#region IRenderingContext implementation
		IEcoServiceProvider IRenderingContext.ServiceProvider
		{
			get { return m_Handle; }
		}
		#endregion

		#region ITupleRenderingContext implementation
		public RenderedTupleDescriptor TupleDescriptor
		{
			get { return m_TupleDescriptor; }
		}

		void ITupleRenderingContext.PartOutOfDate()
		{
			Enqueue();
		}
		#endregion

		#region ISubscribableElementProvider implementation
		IElement IElementProvider.Element
		{
			get { return m_Handle.Element; }
		}

		void ISubscribableElementProvider.SubscribeToElement(ISubscriber subscriber)
		{
			m_Handle.SubscribeToElement(subscriber);
		}
		#endregion
	}

	public sealed class TupleContext: ITupleRenderingContext
	{
		private ITupleRenderingContext m_Owner;

		public TupleContext(ITupleRenderingContext owner)
		{
			m_Owner = owner;
		}

		#region IRenderingContext implementation
		IEcoServiceProvider IRenderingContext.ServiceProvider
		{
			get { return m_Owner.ServiceProvider; }
		}
		#endregion

		#region ITupleRenderingContext implementation
		public RenderedTupleDescriptor TupleDescriptor
		{
			get { return m_Owner.TupleDescriptor; }
		}

		void ITupleRenderingContext.PartOutOfDate()
		{
		}
		#endregion
	}

	public sealed class RenderedList: IList, ITypedList, IBindingList
	{
		private readonly ITupleListRenderingContext m_Context;
		private RenderedListContents m_List;
		private PropertyDescriptor m_SortProperty; // Not used yet.
		private ListSortDirection m_SortDirection = ListSortDirection.Ascending;

		public ITupleListRenderingContext Context
		{
			get { return m_Context; }
		}

		public void MakeListsCurrent()
		{
			GetEnsuredList().MakeListsCurrent();
		}

		public void TupleDescriptorChanged()
		{
			GetEnsuredList().TupleDescriptorChanged();
		}

		internal void SignalListChanged(ListChangedEventArgs args)
		{
			if (m_ListChanged != null)
				m_ListChanged(this, args);
		}

		private RenderedListContents GetEnsuredList()
		{
			if (m_List == null)
			{
				m_List = new RenderedListContents(this);
				m_List.EnsureCurrent();
			}
			return m_List;
		}

		private RenderedListContents GetList()
		{
			return GetEnsuredList();
		}

		private IElementCollection ElementCollection
		{
			get { return m_Context.Element == null ? null : m_Context.Element.GetAsCollection(); }
		}

		private int Count
		{
			get { return GetList().List.Count; }
		}

		#region IEnumerable implementation
		IEnumerator IEnumerable.GetEnumerator()
		{
			return GetList().List.GetEnumerator();
		}
		#endregion

		#region ICollection implementation
		int ICollection.Count
		{
			get { return Count; }
		}

		bool ICollection.IsSynchronized
		{
			get { return false; }
		}

		object ICollection.SyncRoot
		{
			get { return this; }
		}

		void ICollection.CopyTo(System.Array array, int index)
		{
			GetList().List.CopyTo(array, index);
		}
		#endregion

		#region IList implementation
		bool IList.IsFixedSize
		{
			get { return true; }
		}

		bool IList.IsReadOnly
		{
			get { return true; }
		}

		public object this[int index]
		{
			get { return GetList().List[index]; }
			set { GetList().List[index] = value; }
		}

		///<exception cref="NotSupportedException">Thrown always.</exception>
		int IList.Add(object value)
		{
			throw new NotSupportedException();
		}

		///<exception cref="NotSupportedException">Thrown if the collection is read only.</exception>
		void IList.Clear()
		{
			if ((ElementCollection == null) || ElementCollection.ReadOnly)
				throw new NotSupportedException();
			ElementCollection.Clear();
		}

		bool IList.Contains(object value)
		{
			return GetList().List.Contains(value);
		}

		int IList.IndexOf(object value)
		{
			return GetList().List.IndexOf(value);
		}

		///<exception cref="NotSupportedException">Thrown always.</exception>
		void IList.Insert(int index, object value)
		{
			throw new NotSupportedException();
		}

		///<exception cref="NotSupportedException">Thrown if the collection does not support RemoveAt.</exception>
		private void RemoveAt(int index)
		{
			if ((ElementCollection == null) || (!ElementCollection.SupportsRemoveAt))
				throw new NotSupportedException();
			if (index != -1)
			{
				ElementCollection.RemoveAt(index);
				GetList().RemoveAt(index);
			}
		}

		void IList.Remove(object value)
		{
			int index = GetList().List.IndexOf(value);
			RemoveAt(index);
		}

		void IList.RemoveAt(int index)
		{
			RemoveAt(index);
		}
		#endregion

		#region ITypedList implementation
		PropertyDescriptorCollection ITypedList.GetItemProperties(PropertyDescriptor[] listAccessors)
		{
			if ((listAccessors == null) || (listAccessors.Length == 0))
				return m_Context.TupleDescriptor;
			else if (listAccessors[0] is RenderedTuplePropertyDescriptor)
				return ((RenderedTuplePropertyDescriptor)listAccessors[0]).GetItemProperties(listAccessors, 1);
			else
				return PropertyDescriptorCollection.Empty;
		}

		string ITypedList.GetListName(PropertyDescriptor[] listAccessors)
		{
			if ((listAccessors == null) || (listAccessors.Length == 0))
				return string.Empty;
			RenderedTuplePropertyDescriptor rtpd = listAccessors[listAccessors.Length-1] as RenderedTuplePropertyDescriptor;
			if (rtpd != null)
				return rtpd.Name;
			return string.Empty;
		}
		#endregion

		#region IBindingList implementation
		bool IBindingList.AllowEdit
		{
			get { return m_Context.TupleDescriptor.HasEditableProperty; }
		}

		bool IBindingList.AllowNew
		{
			// Don't try to implement again. Surfaces to many problems in the
			// MS implementaion of databinding, especially in the DataGrid
			get { return false; }
		}

		bool IBindingList.AllowRemove
		{
			get
			{
				return ElementCollection == null ? false : !ElementCollection.ReadOnly; // FIXME check statically?
			}
		}

		public bool IsSorted
		{
			get { return m_SortProperty != null; }
		}

		public ListSortDirection SortDirection
		{
			get { return m_SortDirection; }
		}

		public PropertyDescriptor SortProperty
		{
			get { return m_SortProperty; }
		}

		bool IBindingList.SupportsChangeNotification
		{
			get { return true; }
		}

		bool IBindingList.SupportsSearching
		{
			get { return true; }
		}

		bool IBindingList.SupportsSorting
		{
			get { return false; }
		}

		void IBindingList.AddIndex(PropertyDescriptor property)
		{
			// FIXME could implement, but is it really important?
		}

		///<exception cref="NotSupportedException">Thrown always.</exception>
		object IBindingList.AddNew()
		{
			throw new NotSupportedException();
		}

		void IBindingList.ApplySort(PropertyDescriptor property, ListSortDirection direction)
		{
			m_SortProperty = property;
			m_SortDirection = direction;
			// FIXME notifiy contents
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="property"/> is null.</exception>
		int IBindingList.Find(PropertyDescriptor property, object key)
		{
			if (property == null) throw new ArgumentNullException("property"); // do not localize
			for (int i = 0; i < Count; i++)
				if (property.GetValue(this[i]).Equals(key))
					return i;
			return -1;
		}

		void IBindingList.RemoveIndex(PropertyDescriptor property)
		{
		}

		void IBindingList.RemoveSort()
		{
			m_SortProperty = null;
		}

		private event ListChangedEventHandler m_ListChanged;
		event ListChangedEventHandler IBindingList.ListChanged
		{
			add {m_ListChanged += value;}
			remove {m_ListChanged -= value;}
		}
		#endregion

		public RenderedList(ITupleListRenderingContext context): base()
		{
			m_Context = context;
		}
	}
	/// <summary>
	/// This class is part of the internal implementation of the databindable Ilist
	/// implemented by the Eco handles. It should not be used directly.
	/// </summary>
	public sealed class RenderedTupleDescriptor: PropertyDescriptorCollection
	{
		private readonly IStaticContext m_StaticContext;
		private DescriptorFactory m_DescriptorFactory;

		public DescriptorFactory DescriptorFactory
		{
			get { return m_DescriptorFactory; }
		}

		public IStaticContext StaticContext
		{
			get { return m_StaticContext; }
		}

		public RenderedTupleDescriptor(IStaticContext staticContext): base(null)
		{
			m_StaticContext = staticContext;
		}

		public event DeriveEventHandler DeriveValue;
		public event ReverseDeriveEventHandler ReverseDeriveValue;

		public event SignalListChangedHandler DescriptorsChanged;

		public bool IsDeriveValueAssigned
		{
			get { return DeriveValue != null; }
		}

		public bool IsReverseDeriveValueAssigned
		{
			get { return ReverseDeriveValue != null; }
		}

		internal void ExecuteDeriveValue(DeriveEventArgs e)
		{
			DeriveValue(this, e);
		}

		internal void ExecuteReverseDeriveValue(ReverseDeriveEventArgs e)
		{
			ReverseDeriveValue(this, e);
		}

		private bool NonExistingPropDesc(string aName)
		{
			foreach(RenderedTuplePropertyDescriptor x in this)
			{
				if (string.Compare(x.Name, aName, true, CultureInfo.CurrentCulture) == 0) // case insensitive
					return false;
			}
			return true;
		}

		public bool HasEditableProperty
		{
			get
			{
				foreach (RenderedTuplePropertyDescriptor d in this)
					if (!d.IsReadOnly)
						return true;
				return false;
			}
		}

		private void AddDescriptor(RenderedTuplePropertyDescriptor rtpd, bool designMode)
		{
			// Don't add list types at design time, since designer tries to do AddNew() to
			// analyze them
			if (designMode && (rtpd.IsNested || rtpd.PropertyType.IsArray))
				return;
			Add(rtpd);
			if (DescriptorsChanged != null)
				DescriptorsChanged(new ListChangedEventArgs(ListChangedType.PropertyDescriptorAdded, rtpd));
		}

		private static bool ShowMember(VisibilityKind visibility, MemberVisibility memberVisibility)
		{
			if (visibility == VisibilityKind.Private_ && !(memberVisibility == MemberVisibility.AllMembers)) return false;

			if (visibility == VisibilityKind.Protected_ &&
				!((memberVisibility == MemberVisibility.ProtectedOrHigher) ||
					(memberVisibility == MemberVisibility.AllMembers))) return false;
			return true;
		}

		internal void SetupProperties(ColumnCollection columns, bool addDefaultProperties, bool addDefaultNestings, bool addExternalId, MemberVisibility memberVisibility, DescriptorFactory descriptorFacory, bool designMode)
		{
			m_DescriptorFactory = descriptorFacory;
			// Always redo everything

			while (Count > 0)
			{
				RenderedTuplePropertyDescriptor last = this[Count - 1] as RenderedTuplePropertyDescriptor;
				RemoveAt(Count - 1);
				if (DescriptorsChanged != null)
					DescriptorsChanged(new ListChangedEventArgs(ListChangedType.PropertyDescriptorDeleted, last));
			}

			if (columns != null)
				foreach(AbstractColumn aColumn in columns)
					AddDescriptor(new RenderedTuplePropertyDescriptor(this, aColumn), designMode);
			if ((addDefaultProperties || addDefaultNestings || addExternalId) && m_StaticContext != null &&
				m_StaticContext.StaticUmlType != null)
			{
				IClassifier c = m_StaticContext.StaticUmlType;
				if (c is ICollectionType)
					throw new NotSupportedException("Internal error"); // Do not localize
				if (c is IClass)
				{
					if (addExternalId && NonExistingPropDesc("ExternalId")) // Do not localize
						AddDescriptor(new RenderedTuplePropertyDescriptor("ExternalId", this, "self.externalId", false), designMode); // Do not localize

					IClass ci = (IClass)c;
					for (int i = 0; i <= ci.EcoClass.AllStructuralFeatures.Count - 1; i++)
					{
						IStructuralFeature sf = ci.EcoClass.AllStructuralFeatures[i];

						if (!ShowMember(sf.Visibility, memberVisibility)) continue;

						if ((sf is IAttribute ) && addDefaultProperties) // && !sf.Type_.ObjectType.IsArray) - breaks autoforms
						{
							IAttribute attr = (IAttribute)sf;
							if (NonExistingPropDesc(attr.Name))
								AddDescriptor(new RenderedTuplePropertyDescriptor(attr.Name, this, "self." + attr.Name, false), designMode); // Do not localize
						}
						else if ((sf is IAssociationEnd))
						{
							IAssociationEnd role = (IAssociationEnd)sf;
							if(((role.Name).Length > 0) && role.IsNavigable)
							{
								if (addDefaultNestings)
									AddDescriptor(new RenderedTuplePropertyDescriptor(role.Name, this, "self." + role.Name, true), designMode); // Do not localize
								else if (addDefaultProperties && role.Multiplicity.Upper == 1)
									AddDescriptor(new RenderedTuplePropertyDescriptor(role.Name, this, "self." + role.Name + ".asString", false), designMode); // Do not localize
							}
						}
					}
				}
				else if (c is IPrimitiveType && NonExistingPropDesc("self"))
					AddDescriptor(new RenderedTuplePropertyDescriptor("self", this, "self", false), designMode); // Do not localize
			}
		}
	}

	public sealed class RenderedTuplePropertyDescriptor: PropertyDescriptor
	{
		private readonly RenderedTupleDescriptor m_OwningTupleDescriptor;
		private readonly string m_Expression;
		private readonly AbstractColumn m_Column;
		private readonly bool m_IsNested;
		private readonly bool m_IsEventDerived;
		private readonly string m_NestingName;
		private readonly IClassifier m_StaticType;
		private readonly bool m_Readonly;

		internal RenderedTuplePropertyDescriptor(RenderedTupleDescriptor owner, AbstractColumn column): base(column.Name, null)
		{
			m_Column = column;
			m_OwningTupleDescriptor = owner;
			m_Expression = column.Expression;
			m_IsNested = column.Nested;
			m_IsEventDerived = column.EventDerivedValue;
			m_NestingName = column.NestingName;
			EventDerivedColumn ec = column as EventDerivedColumn;
			if (ec != null)
			{
				m_Readonly = column.IsReadOnly;
				IEcoTypeSystem typeSystem = m_OwningTupleDescriptor.StaticContext.TypeSystem;
				m_StaticType = typeSystem != null ? typeSystem.GetClassifierByName(ec.TypeName) : null;
			}
			else
			{
				m_Readonly = column.IsReadOnly | GetReadOnly(m_OwningTupleDescriptor.StaticContext, m_Expression);
				m_StaticType = GetStaticType(m_OwningTupleDescriptor.StaticContext, m_Expression);
			}
		}

		internal RenderedTuplePropertyDescriptor(string name, RenderedTupleDescriptor owner, string expression, bool nested): base(name, null)
		{
			m_OwningTupleDescriptor = owner;
			m_Expression = expression;
			m_IsNested = nested;

			m_Readonly = GetReadOnly(m_OwningTupleDescriptor.StaticContext, m_Expression);
			m_StaticType = GetStaticType(m_OwningTupleDescriptor.StaticContext, m_Expression);
			if (IsNested && (m_StaticType != null))
			{
				IClassifier NameProviderClassifier = m_StaticType is ICollectionType ? ((ICollectionType)m_StaticType).ElementType : m_StaticType;
				m_NestingName = NameProviderClassifier.Name;
			}
			else
				m_NestingName = "<none>"; // Improbable name, otherwise "" will bind to default on unknown type.
		}

		private bool GetReadOnly(IStaticContext staticContext, string expression)
		{

			try
			{
				IOclTypeService oclTypeService = m_OwningTupleDescriptor.StaticContext.OclTypeService;
				if (oclTypeService == null)
					return false;

				return oclTypeService.ExpressionIsReadOnly(m_Expression,
					m_OwningTupleDescriptor.StaticContext.StaticUmlType,
					false,
					m_OwningTupleDescriptor.StaticContext.VariableList);
			}
			catch
			{
				return false; // Ignore all OCL errors
			}
		}

		private IClassifier GetStaticType(IStaticContext staticContext, string expression)
		{
			try
			{
				IOclTypeService oclTypeService = m_OwningTupleDescriptor.StaticContext.OclTypeService;
				if (oclTypeService == null)
					return null;
				return oclTypeService.ExpressionType(m_Expression,
					m_OwningTupleDescriptor.StaticContext.StaticUmlType,
					false,
					m_OwningTupleDescriptor.StaticContext.VariableList);
				}
			catch
			{
				return null; // Ignore all OCL errors
			}
		}

		public Boolean IsEventDerived
		{
			get { return m_IsEventDerived; }
		}

		public string NestingName
		{
			get { return m_NestingName; }
		}

		public string Expression
		{
			get { return m_Expression; }
		}

		private IClassifier StaticType
		{
			get { return m_StaticType; }
		}

		#region PropertyDescriptor Overrides

		private AbstractRenderedTupleProperty GetTupleProperty(object component)
		{
			if (component == null)
				throw new ArgumentNullException("component");
			AbstractRenderedTupleProperty p = ((AbstractRenderedTuple)component).GetProperty(this);
			if (p == null)
				throw new ArgumentException("component");
			return p;
		}
		public override object GetValue(object component)
		{
			return GetTupleProperty(component).GetValue();
		}

		public override void SetValue(object component, object value)
		{
			GetTupleProperty(component).SetValue(value);
		}

		public override System.Type ComponentType
		{
			get { return typeof(System.Object); }
		}

		public override bool IsReadOnly
		{
			get { return m_Readonly; }
		}

		public override System.Type PropertyType
		{
			get
			{
				if (IsNested)
					return (typeof(IList));
				System.Type result = null;
				if (StaticType != null)
				{
					if (StaticType.ClassifierType == ClassifierType.Collection)
						result = typeof(System.Int32); // Collections are represented as their Count, at the moment
					else
						result = StaticType.ObjectType;
				}
				if (result == null)
					result = typeof(System.Object);
				return result;
			}
		}

		public override bool CanResetValue(object component)
		{
			return false;
		}

		public override void ResetValue(object component)
		{
		}

		public override bool ShouldSerializeValue(object component)
		{
			return false;
		}
		#endregion

		// Nested descriptor
		public bool IsNested
		{
			get { return m_IsNested; }
		}

		private RenderedTupleDescriptor m_ItemProperties;

		public RenderedTupleDescriptor GetOwnItemProperties()
		{
			if (!IsNested)
				return null;
			// lazy create, can go to any depth
			if (m_ItemProperties == null)
				m_ItemProperties = m_OwningTupleDescriptor.DescriptorFactory.GetDescriptor(HandleUtils.ItemType(StaticType), m_NestingName);
			return m_ItemProperties;
		}

		public RenderedTupleDescriptor GetItemProperties(PropertyDescriptor[] listAccessors, int startIndex)
		{
			if ((listAccessors.Length == startIndex))
				return GetOwnItemProperties();
			else if (listAccessors[startIndex] is RenderedTuplePropertyDescriptor)
				return ((RenderedTuplePropertyDescriptor)listAccessors[startIndex]).GetItemProperties(listAccessors, startIndex + 1);
			return null;
		}

		// Rendering
		///<exception cref="ArgumentNullException">Thrown if <paramref name="serviceProvider"/> is null.</exception>
		public IElement DeriveAndSubscribe(IEcoServiceProvider serviceProvider, IElement rootElement, ISubscriber valueChangeSubscriber, ISubscriber resubscribeSubscriber)
		{
			if (serviceProvider == null) throw new ArgumentNullException("serviceProvider");
			IOclService oclService = serviceProvider.GetEcoService(typeof(IOclService))as IOclService;
			if (IsEventDerived && m_OwningTupleDescriptor.IsDeriveValueAssigned)
			{

				DeriveEventArgs e = new DeriveEventArgs(Name, serviceProvider,
					rootElement,
					valueChangeSubscriber,
					resubscribeSubscriber, m_OwningTupleDescriptor.StaticContext);
				m_OwningTupleDescriptor.ExecuteDeriveValue(e);
				return e.ResultElement;
			}
			else
			{
				if ((oclService != null) && (m_OwningTupleDescriptor.StaticContext != null))
					return oclService.EvaluateAndSubscribe(rootElement, m_OwningTupleDescriptor.StaticContext.VariableList, m_Expression, valueChangeSubscriber, resubscribeSubscriber);
				else
					return null;
			}
		}

		public void ReverseDerive(IEcoServiceProvider serviceProvider, IElement rootElement, object value)
		{
			if (IsEventDerived && m_OwningTupleDescriptor.IsDeriveValueAssigned)
			{

				ReverseDeriveEventArgs e = new ReverseDeriveEventArgs(Name, serviceProvider,
					rootElement,
					m_OwningTupleDescriptor.StaticContext, value);
				m_OwningTupleDescriptor.ExecuteReverseDeriveValue(e);
			}
		}

		// Change callback
		internal void DoOnValueChanged(Object o, EventArgs e)
		{
			base.OnValueChanged(o, e);
		}
	}

	public sealed class PropertyComparer: IComparer
	{
		private readonly PropertyDescriptor m_PropertyDescriptor;

		public PropertyComparer(PropertyDescriptor propertyDescriptor)
		{
			m_PropertyDescriptor = propertyDescriptor;
		}

		int IComparer.Compare(object x, object y)
		{
			IComparable comparableX = m_PropertyDescriptor.GetValue(x) as IComparable;
			if (comparableX == null)
				return 0; // silently regard as equal
			return comparableX.CompareTo(m_PropertyDescriptor.GetValue(y));
		}
	}

	public sealed class RenderedListContents: AbstractDeriver
	{
		private readonly ArrayList m_ItemList = new ArrayList(); // of RenderedTuple
		private readonly RenderedList m_Owner;
		private bool m_TupleDescriptorChanged;

		public override string ToString()
		{
			return m_Owner.Context.Element == null ? base.ToString() : m_Owner.Context.Element.AsObject.ToString();
		}

		public void TupleDescriptorChanged()
		{
			m_TupleDescriptorChanged = true;
		}

		public IList List
		{
			get { return m_ItemList; }
		}

		public void RemoveAt(int index)
		{
			m_ItemList.RemoveAt(index);
			m_Owner.SignalListChanged(new ListChangedEventArgs(ListChangedType.ItemDeleted, index));
		}

		public void MakeListsCurrent()
		{
			EnsureCurrent();
			for (int i = 0; i < m_ItemList.Count; i++)
				if (this[i].HasUnNotifiedChange)
				{
					m_Owner.SignalListChanged(new ListChangedEventArgs(ListChangedType.ItemChanged, i));
					this[i].MakeListsCurrent();
				}
		}

		private AbstractRenderedTuple this[int index]
		{
			get { return (AbstractRenderedTuple)m_ItemList[index]; }
		}

		public RenderedListContents(RenderedList context)
		{
			m_Owner = context;
		}

		protected override void DoNotifyOutOfDate()
		{
			m_Owner.Context.PartOutOfDate();
		}

		private static ListChangedEventArgs MergeEvents(ListChangedEventArgs oldEvent, ListChangedEventArgs newEvent)
		{
			if (oldEvent == null)
				return newEvent;
			if (newEvent == null)
				return oldEvent;
			return new ListChangedEventArgs(ListChangedType.Reset, -1);
		}

		private ListChangedEventArgs EnsureElement(int index, IElement e, ListChangedEventArgs oldEvent)
		{
			if ((m_ItemList.Count > (index)) && (this[index]).Element.IsEqual(e))
				return oldEvent;
			// Try single remove
			if ((m_ItemList.Count > (index + 1)) && this[index + 1].Element.IsEqual(e))
			{
				m_ItemList.RemoveAt(index);
				return MergeEvents(oldEvent, new ListChangedEventArgs(ListChangedType.ItemDeleted, index));
			}
			// insert, trim later
			m_ItemList.Insert(index, new RenderedTuple(e, m_Owner.Context));
			return MergeEvents(oldEvent, new ListChangedEventArgs(ListChangedType.ItemAdded, index));
		}

		private ListChangedEventArgs Trim(int trimmedCount, ListChangedEventArgs oldEvent)
		{
			ListChangedEventArgs newEvent = oldEvent;
			while (m_ItemList.Count > trimmedCount)
			{
				m_ItemList.RemoveAt(m_ItemList.Count - 1);
				newEvent = MergeEvents(newEvent, new ListChangedEventArgs(ListChangedType.ItemDeleted, m_ItemList.Count));
			}
			return newEvent;
		}

		protected override void DoDeriveAndSubscribe(bool subscribe)
		{
			ListChangedEventArgs args = null;
			if (m_TupleDescriptorChanged)
			{
				m_ItemList.Clear();
				args = new ListChangedEventArgs(ListChangedType.Reset, -1);
			}
			IElement e = m_Owner.Context.Element;
			if (e == null ||
				((e.ContentType == ContentType.Object) && ((IObject)e).Count == 0))
				args = Trim(0, args);
			else
			{
				IElementCollection list = m_Owner.Context.Element as IElementCollection;
				if (list == null)
				{
					args = EnsureElement(0, e, args);
					args = Trim(1, args);
				}
				else
				{
					for (int i = 0; i < list.Count; i++)
						args = EnsureElement(i, list[i], args);
					args = Trim(list.Count, args);
				}
			}

			if (subscribe)
			{
				m_Owner.Context.SubscribeToElement(ResubscribeSubscriber);
				if (m_Owner.Context.Element != null)
					m_Owner.Context.Element.SubscribeToValue(ReevaluateSubscriber);
			}
			if (args != null)
				m_Owner.SignalListChanged(args);
			m_TupleDescriptorChanged = false;
		}

		public void Deactivate()
		{
			CancelAllSubscriptions();
		}
	}

	public sealed class RenderedTuple: AbstractRenderedTuple
	{
		public RenderedTuple(IElement element, ITupleRenderingContext owner): base(element, owner)
		{
		}
	}

	// This class not used yet
	public sealed class EditableRenderedTuple: AbstractRenderedTuple, IEditableObject
	{
		public EditableRenderedTuple(IElement element, ITupleRenderingContext owner): base(element, owner)
		{
		}
		#region IRenderingContext implementation
		void IEditableObject.BeginEdit()
		{
		}
		void IEditableObject.EndEdit()
		{
		}
		void IEditableObject.CancelEdit()
		{
			IObject o = Element as IObject;
			if (o != null)
				o.Delete();
		}
		#endregion
	}
	/// <summary>
	/// This class is part of the internal implementation of the databindable Ilist
	/// implemented by the Eco handles. It should not be used directly.
	/// </summary>
	public abstract class AbstractRenderedTuple: ICustomTypeDescriptor, ITuplePropertyRenderingContext
	{
		private bool m_NeedsPush;
		private	AbstractRenderedTupleProperty[] m_Properties;
		private readonly ITupleRenderingContext m_Owner;
		private readonly IElement m_Root;

		internal bool HasUnNotifiedChange
		{
			get { return m_NeedsPush; }
		}

		internal void MakeListsCurrent()
		{
			foreach (AbstractRenderedTupleProperty rtp in m_Properties)
				if (rtp != null)
					rtp.MakeListsCurrent();
			m_NeedsPush = false;
		}

		public override bool Equals(object obj)
		{
			AbstractRenderedTuple rt = obj as AbstractRenderedTuple;
			if (rt == null)
				return false;
			return (m_Root.IsEqual(rt.m_Root) && (m_Owner.TupleDescriptor == rt.m_Owner.TupleDescriptor));
		}

		public override int GetHashCode()
		{
			return m_Root.GetIsEqualHashCode();
		}

		#region IRenderingContext implementation
		IEcoServiceProvider IRenderingContext.ServiceProvider
		{
			get { return m_Owner.ServiceProvider; }
		}
		#endregion

		#region ITuplePropertyRenderingContext implementation
		void ITuplePropertyRenderingContext.PropertyOutOfDate()
		{
			m_NeedsPush = true;
			m_Owner.PartOutOfDate();
		}

		public IElement Element // Doubles for ITuplePropertyRenderingContext
		{
			get { return m_Root; }
		}
		#endregion

		#region ICustomTypeDescriptor implementation
		AttributeCollection ICustomTypeDescriptor.GetAttributes()
		{
			return AttributeCollection.Empty;
		}

		string ICustomTypeDescriptor.GetClassName()
		{
			return null;
		}

		string ICustomTypeDescriptor.GetComponentName()
		{
			return null;
		}

		TypeConverter ICustomTypeDescriptor.GetConverter()
		{
			return null;
		}

		EventDescriptor ICustomTypeDescriptor.GetDefaultEvent()
		{
			return null;
		}

		PropertyDescriptor ICustomTypeDescriptor.GetDefaultProperty()
		{
			return m_Owner.TupleDescriptor.Count > 0 ? m_Owner.TupleDescriptor[0] : null;
		}

		object ICustomTypeDescriptor.GetEditor(Type editorBaseType)
		{
			return null;
		}

		EventDescriptorCollection ICustomTypeDescriptor.GetEvents()
		{
			return EventDescriptorCollection.Empty;
		}

		EventDescriptorCollection ICustomTypeDescriptor.GetEvents(Attribute[] attributes)
		{
			return EventDescriptorCollection.Empty;
		}

		PropertyDescriptorCollection ICustomTypeDescriptor.GetProperties()
		{
			return m_Owner.TupleDescriptor;
		}

		PropertyDescriptorCollection ICustomTypeDescriptor.GetProperties(Attribute[] attributes)
		{
			return m_Owner.TupleDescriptor;
		}

		object ICustomTypeDescriptor.GetPropertyOwner(PropertyDescriptor pd)
		{
			return this;
		}
		#endregion

		///<exception cref="ArgumentException">Thrown if <paramref name="descriptor"/> does not exist in the list of the owner's descriptors.</exception>
		public AbstractRenderedTupleProperty GetProperty(RenderedTuplePropertyDescriptor descriptor)
		{
			if (m_Properties == null)
				m_Properties = new AbstractRenderedTupleProperty[m_Owner.TupleDescriptor.Count];
			int index = m_Owner.TupleDescriptor.IndexOf(descriptor);
			if (index == -1)
				throw new ArgumentException(HandlesStringRes.sDescriptorIsNotAProperty);

			if (m_Properties[index] == null)
			{
				if (descriptor.IsNested)
					m_Properties[index] = new RenderedTupleListProperty(descriptor, this);
				else
					m_Properties[index] = new RenderedTupleProperty(descriptor, this);
			}
			return m_Properties[index];
		}

		protected AbstractRenderedTuple(IElement element, ITupleRenderingContext owner)
		{
			m_Root = element;
			m_Owner = owner;
		}
	}

	public abstract class AbstractRenderedTupleProperty: AbstractDeriver, ISubscribableElementProvider
	{
		protected IElement m_Value;
		protected readonly RenderedTuplePropertyDescriptor m_Descriptor;
		protected readonly ITuplePropertyRenderingContext m_Owner;
		private readonly Publisher m_elementChangedPublisher = new Publisher();

		protected IElement Value
		{
			get
			{
				EnsureCurrent();
				return m_Value;
			}
		}

		protected AbstractRenderedTupleProperty(RenderedTuplePropertyDescriptor descriptor, ITuplePropertyRenderingContext owner)
		{
			m_Descriptor = descriptor;
			m_Owner = owner;
		}

		internal virtual void MakeListsCurrent()
		{
		}

		public override string ToString()
		{
			return m_Owner.Element == null ? base.ToString() : m_Owner.Element.AsObject.ToString();
		}
		#region AbstractDeriver overrides
		protected override void DoNotifyOutOfDate()
		{
			m_Owner.PropertyOutOfDate();
			m_elementChangedPublisher.Send(this, EventArgs.Empty);
		}

		protected override void DoDeriveAndSubscribe(bool subscribe)
		{
			IElement oldValue = m_Value;

			if (subscribe)
				m_Value = m_Descriptor.DeriveAndSubscribe(m_Owner.ServiceProvider, m_Owner.Element, ReevaluateSubscriber, ResubscribeSubscriber);
			else
				m_Value = m_Descriptor.DeriveAndSubscribe(m_Owner.ServiceProvider, m_Owner.Element, null, null);

			if ((oldValue != null) && !oldValue.Equals(m_Value)) // Don't send first time
				m_Descriptor.DoOnValueChanged(this, null);
		}
		#endregion

		#region ISubscribableElementProvider implementation
		IElement IElementProvider.Element
		{
			get { return Value; }
		}
		void ISubscribableElementProvider.SubscribeToElement(ISubscriber subscriber)
		{
			m_elementChangedPublisher.AddSubscription(subscriber);
		}
		#endregion

		public abstract object GetValue();
		public abstract void SetValue(object value);
	}

	public sealed class RenderedTupleProperty: AbstractRenderedTupleProperty
	{

		public RenderedTupleProperty(RenderedTuplePropertyDescriptor descriptor,
			ITuplePropertyRenderingContext owner): base(descriptor, owner)
		{
		}

		public override object GetValue()
		{
			if (Value == null)
				return DBNull.Value;
			if (Value is IElementCollection)
				return ((IElementCollection)Value).Count;
			if (Value is IObjectList)
				return ((IObjectList)Value).Count;

			System.Object res = Value.AsObject;
			return res == null ? DBNull.Value : res;
		}
		public override void SetValue(object value)
		{
			if (value == DBNull.Value)
				value = null;

			if (m_Descriptor.IsEventDerived)
				m_Descriptor.ReverseDerive(m_Owner.ServiceProvider, m_Owner.Element, value);
			else if ((Value != null) && Value.Mutable)
				m_Value.AsObject = value;
		}
	}

	public sealed class RenderedTupleListProperty: AbstractRenderedTupleProperty, ITupleListRenderingContext
	{
		private RenderedList m_List;

		public RenderedTupleListProperty(RenderedTuplePropertyDescriptor descriptor,
			ITuplePropertyRenderingContext owner): base(descriptor, owner)
		{
		}

		internal override void MakeListsCurrent()
		{
			if (m_List != null)
				m_List.MakeListsCurrent();
		}

		public override object GetValue()
		{
			if (m_List == null)
			{
				m_List = new RenderedList(this);
				MakeListsCurrent();
			}
			return m_List;
		}

		///<exception cref="InvalidOperationException">Thrown always.</exception>
		public override void SetValue(object value)
		{
			throw new InvalidOperationException();
		}

		#region IRenderingContext implementation
		IEcoServiceProvider IRenderingContext.ServiceProvider
		{
			get { return m_Owner.ServiceProvider; }
		}

		#endregion

		#region ITupleRenderingContext implementation

		RenderedTupleDescriptor ITupleRenderingContext.TupleDescriptor
		{
			get { return m_Descriptor.GetOwnItemProperties(); }
		}
		void ITupleRenderingContext.PartOutOfDate()
		{
			m_Owner.PropertyOutOfDate();
		}
		#endregion
	}

	public sealed class DescriptorFactory
	{
		private readonly Hashtable m_Descriptors = new Hashtable();
		private readonly NestingCollection m_Nestings;
		private readonly IStaticContext m_staticContext;
		private readonly bool m_AddDefaultNestings;
		private readonly bool m_DesignMode;
		private readonly MemberVisibility m_MemberVisibility;

		public DescriptorFactory(IStaticContext staticContext, NestingCollection nestings, Boolean addDefaultNestings, MemberVisibility memberVisibility, bool designMode)
		{
			m_staticContext = staticContext;
			m_Nestings = nestings;
			m_AddDefaultNestings = addDefaultNestings;
			m_DesignMode = designMode;
			m_MemberVisibility = memberVisibility;
		}

		public RenderedTupleDescriptor GetDescriptor(IClassifier staticType, string nestingName)
		{
			DescriptorKey key = new DescriptorKey(staticType, nestingName);
			RenderedTupleDescriptor result = m_Descriptors[key] as RenderedTupleDescriptor;
			if (result == null)
			{
				result = new RenderedTupleDescriptor(new FixedTypeStaticContext(m_staticContext, staticType));
				Nesting n = m_Nestings[nestingName];

				if (n != null)
					result.SetupProperties(n.Columns, n.AddDefaultProperties, n.AddDefaultNestings, false, m_MemberVisibility, this, m_DesignMode);
				else
					result.SetupProperties(null, true, m_AddDefaultNestings, false, MemberVisibility.PublicOnly, this, m_DesignMode);
				m_Descriptors.Add(key, result);

			}
			return result;
		}

		private sealed class DescriptorKey
		{
			IClassifier m_StaticType;
			string m_NestingName;

			public DescriptorKey(IClassifier staticType, string nestingName)
			{
				m_StaticType = staticType;
				m_NestingName = nestingName;
				if (nestingName == null || nestingName.Length == 0)
					m_NestingName = staticType.Name;
			}

			public override bool Equals(object obj)
			{
				DescriptorKey d = (DescriptorKey)obj;
				return m_StaticType.IsSame(d.m_StaticType) && (m_NestingName == d.m_NestingName);
			}

			public override int GetHashCode()
			{
				return m_StaticType.GetIsSameHashCode() ^ m_NestingName.GetHashCode();
			}
		}

		private sealed class FixedTypeStaticContext: IStaticContext
		{
			private readonly IStaticContext m_AdaptedContext;
			private readonly IClassifier m_StaticType;

			public FixedTypeStaticContext(IStaticContext adaptedContext, IClassifier staticType)
			{
				m_AdaptedContext = adaptedContext;
				m_StaticType = staticType;
			}

			IEcoTypeSystem IStaticContext.TypeSystem
			{
				get { return m_AdaptedContext.TypeSystem; }
			}

			IOclTypeService IStaticContext.OclTypeService
			{
				get { return m_AdaptedContext.OclTypeService; }
			}

			IOclPsTypeService IStaticContext.OclPsTypeService
			{
				get { return m_AdaptedContext.OclPsTypeService; }
			}

			IActionLanguageTypeService IStaticContext.ActionLanguageTypeService
			{
				get { return m_AdaptedContext.ActionLanguageTypeService; }
			}

			IExternalVariableList IStaticContext.VariableList
			{
				get { return m_AdaptedContext.VariableList; }
			}

			IClassifier IStaticContext.StaticUmlType
			{
				get { return m_StaticType; }
			}
		}
	}

	internal abstract class DisplayQueue // Only static methods
	{
		[ThreadStatic] private static ArrayList T_DisplayQueue;
		private readonly static object m_DisplayQueueLock = new object();

		private static ArrayList Queue
		{
			get
			{
				if (T_DisplayQueue == null)
					T_DisplayQueue = new ArrayList();
				return T_DisplayQueue;
			}
		}

		public static void OnIdle(object sender, EventArgs args)
		{
			DisplayAll();
		}

		public static void Remove(INotifyOnIdle displayable)
		{
			lock (m_DisplayQueueLock)
			{
				Queue.Remove(displayable);
			}
		}

		public static void Add(INotifyOnIdle displayable)
		{
			lock (m_DisplayQueueLock)
			{
				Queue.Add(displayable);
			}
		}

		public static void DisplayAll()
		{
			while (Queue.Count > 0)
			{
				ArrayList oldQueue;
				lock (m_DisplayQueueLock)
				{
					oldQueue = Queue;
					T_DisplayQueue = null;
				}
				for (int i = 0; i < oldQueue.Count; i++)
					(oldQueue[i] as INotifyOnIdle).NotifyOnIdle();
			}
		}
	}
}
